/*
 * Decompiled with CFR 0.152.
 */
package org.unicode.cldr.draft.keyboard.out;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.ibm.icu.lang.UCharacter;
import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.UCharacterIterator;
import com.ibm.icu.text.UnicodeSet;
import java.io.Writer;
import org.unicode.cldr.draft.keyboard.CharacterMap;
import org.unicode.cldr.draft.keyboard.IsoLayoutPosition;
import org.unicode.cldr.draft.keyboard.KeyMap;
import org.unicode.cldr.draft.keyboard.Keyboard;
import org.unicode.cldr.draft.keyboard.KeyboardId;
import org.unicode.cldr.draft.keyboard.KeyboardSettings;
import org.unicode.cldr.draft.keyboard.ModifierKeyCombinationSet;
import org.unicode.cldr.draft.keyboard.Transform;
import org.unicode.cldr.draft.keyboard.out.XmlWriter;

public final class KeyboardToXml {
    private final Keyboard keyboard;
    private final XmlWriter xmlWriter;
    private static DecimalFormat VERSION_FORMAT = new DecimalFormat("#.#");
    private static final Joiner COMMA_JOINER = Joiner.on(",");
    private static final UnicodeSet ESCAPED_CHARACTERS_NO_SPACE = new UnicodeSet("[[:di:][:c:][:M:][:whitespace:][\"]-[\\u0020]]").freeze();
    private static final UnicodeSet ILLEGAL_COMMENT_CHARACTERS = new UnicodeSet("[[:di:][:c:][:whitespace:]]").freeze();

    private KeyboardToXml(Keyboard keyboard, XmlWriter xmlWriter) {
        this.keyboard = Preconditions.checkNotNull(keyboard);
        this.xmlWriter = Preconditions.checkNotNull(xmlWriter);
    }

    public static void writeToXml(Keyboard keyboard, Writer writer) {
        XmlWriter xmlWriter = XmlWriter.newXmlWriter(writer);
        KeyboardToXml keyboardToXml = new KeyboardToXml(keyboard, xmlWriter);
        keyboardToXml.toXml();
    }

    private void toXml() {
        this.xmlWriter.startDocument("keyboard", "../dtd/ldmlKeyboard.dtd");
        this.xmlWriter.startElement("keyboard", ImmutableMap.of("locale", this.keyboard.keyboardId()));
        this.addMetadata();
        this.addKeyMaps();
        this.addTransforms();
        this.xmlWriter.endElement();
        this.xmlWriter.endDocument();
    }

    private void addMetadata() {
        ImmutableMap settingsAttributes;
        boolean hasTransform;
        KeyboardId.Platform platform = this.keyboard.keyboardId().platform();
        ImmutableMap<String, String> versionAttributes = ImmutableMap.of("platform", VERSION_FORMAT.format(platform.version()), "number", "$Revision$");
        this.xmlWriter.addElement("version", versionAttributes);
        this.xmlWriter.startElement("names");
        for (String name : this.keyboard.names()) {
            this.xmlWriter.addElement("name", ImmutableMap.of("value", name));
        }
        this.xmlWriter.endElement();
        ImmutableMap.Builder<String, String> settingsBuilder = ImmutableMap.builder();
        if (platform.settings().fallbackSetting() == KeyboardSettings.FallbackSetting.OMIT) {
            KeyboardSettings.FallbackSetting fallbackSetting = platform.settings().fallbackSetting();
            settingsBuilder.put("fallback", fallbackSetting.toString().toLowerCase());
        }
        boolean bl = hasTransform = this.keyboard.transforms().size() > 0;
        if (hasTransform && platform.settings().transformFailureSetting() == KeyboardSettings.TransformFailureSetting.OMIT) {
            KeyboardSettings.TransformFailureSetting transformFailure = platform.settings().transformFailureSetting();
            settingsBuilder.put("transformFailure", transformFailure.toString());
        }
        if (hasTransform && platform.settings().transformPartialSetting() == KeyboardSettings.TransformPartialSetting.HIDE) {
            KeyboardSettings.TransformPartialSetting transformPartial = platform.settings().transformPartialSetting();
            settingsBuilder.put("transformPartial", transformPartial.toString());
        }
        if (!(settingsAttributes = settingsBuilder.build()).isEmpty()) {
            this.xmlWriter.addElement("settings", settingsAttributes);
        }
    }

    private void addKeyMaps() {
        for (KeyMap keyMap : this.keyboard.keyMaps()) {
            ImmutableMap.Builder<String, String> keyMapAttributes = ImmutableMap.builder();
            ModifierKeyCombinationSet modifiers = keyMap.modifierKeyCombinationSet();
            if (!modifiers.isBase()) {
                keyMapAttributes.put("modifiers", modifiers.toString());
            }
            this.xmlWriter.startElement("keyMap", keyMapAttributes.build());
            for (CharacterMap characterMap : keyMap.isoLayoutToCharacterMap().values()) {
                String output = characterMap.output();
                ImmutableMap.Builder<String, String> mapAttributes = ImmutableMap.builder();
                mapAttributes.put("iso", "" + (Object)((Object)characterMap.position()));
                String escapedOutput = this.escapeOutput(output);
                mapAttributes.put("to", escapedOutput);
                if (!characterMap.longPressKeys().isEmpty()) {
                    mapAttributes.put("longPress", COMMA_JOINER.join(characterMap.longPressKeys()));
                }
                if (characterMap.isTransformNo()) {
                    mapAttributes.put("transform", "no");
                }
                String comment = this.buildReadabilityComment(characterMap, escapedOutput);
                this.xmlWriter.addElement("map", mapAttributes.build(), comment);
            }
            this.xmlWriter.endElement();
        }
    }

    private String escapeOutput(String output) {
        int character;
        StringBuilder stringBuilder = new StringBuilder();
        UCharacterIterator it = UCharacterIterator.getInstance(output);
        while ((character = it.nextCodePoint()) != -1) {
            if (ESCAPED_CHARACTERS_NO_SPACE.contains(character)) {
                stringBuilder.append(String.format("\\u{%X}", character));
                continue;
            }
            stringBuilder.append(UCharacter.toString(character));
        }
        return stringBuilder.toString();
    }

    private String buildReadabilityComment(CharacterMap characterMap, String escapedOutput) {
        KeyMap baseMap;
        CharacterMap baseKey;
        StringBuilder comment = new StringBuilder();
        String output = characterMap.output();
        IsoLayoutPosition position = characterMap.position();
        if (!output.toUpperCase().equals(position.englishKeyName())) {
            comment.append(position.englishKeyName());
        }
        if ((baseKey = (baseMap = this.keyboard.baseMap()).isoLayoutToCharacterMap().get((Object)position)) != null && !baseKey.output().toUpperCase().equals(output.toUpperCase()) && !baseKey.output().toUpperCase().equals(position.englishKeyName())) {
            comment.append("  base=");
            comment.append(baseKey.output());
        }
        if (escapedOutput.contains("\\u{") && !ILLEGAL_COMMENT_CHARACTERS.contains(output)) {
            comment.append("  to= " + output + " ");
        }
        return comment.toString();
    }

    private void addTransforms() {
        if (this.keyboard.transforms().isEmpty()) {
            return;
        }
        this.xmlWriter.startElement("transforms", ImmutableMap.of("type", "simple"));
        for (Transform transform : this.keyboard.transforms()) {
            String escapedSequence = this.escapeOutput(transform.sequence());
            String escapedOutput = this.escapeOutput(transform.output());
            String comment = this.buildTransformReadabilityComment(transform, escapedSequence, escapedOutput);
            this.xmlWriter.addElement("transform", ImmutableMap.of("from", escapedSequence, "to", escapedOutput), comment);
        }
        this.xmlWriter.endElement();
    }

    private String buildTransformReadabilityComment(Transform transform, String escapedSequence, String escapedOutput) {
        if ((escapedSequence.contains("\\u{") || escapedOutput.contains("\\u{")) && !ILLEGAL_COMMENT_CHARACTERS.containsSome(transform.sequence()) && !ILLEGAL_COMMENT_CHARACTERS.contains(transform.output())) {
            return transform.sequence() + " \u2192 " + transform.output();
        }
        return "";
    }
}

