/*
 * Decompiled with CFR 0.152.
 */
package org.unicode.cldr.util;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.ibm.icu.impl.Relation;
import com.ibm.icu.impl.Row;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.text.UnicodeSet;
import com.ibm.icu.util.ULocale;
import com.ibm.icu.util.VersionInfo;
import java.text.ParseException;
import java.util.AbstractSet;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Pattern;
import org.unicode.cldr.util.LanguageTagParser;
import org.unicode.cldr.util.Predicate;
import org.unicode.cldr.util.StandardCodes;
import org.unicode.cldr.util.SupplementalDataInfo;
import org.unicode.cldr.util.Validity;

public abstract class MatchValue
implements Predicate<String> {
    public static final String DEFAULT_SAMPLE = "\u2753";
    static final Set<String> SCRIPT_HACK = ImmutableSet.of("Afak", "Blis", "Cirt", "Cyrs", "Egyd", "Egyh", new String[]{"Geok", "Inds", "Jurc", "Kpel", "Latf", "Latg", "Loma", "Maya", "Moon", "Nkgb", "Phlv", "Roro", "Sara", "Syre", "Syrj", "Syrn", "Teng", "Visp", "Wole"});
    static final Set<String> VARIANT_HACK = ImmutableSet.of("POSIX", "REVISED", "SAAHO");
    static final Splitter RANGE = Splitter.on('~').trimResults();
    static final Splitter LIST = Splitter.on(", ").trimResults();
    static final Splitter SPLIT_SPACE_OR_COMMA = Splitter.on(Pattern.compile("[, ]")).omitEmptyStrings().trimResults();
    static final Splitter SPACE_SPLITTER = Splitter.on(' ').omitEmptyStrings();
    static final Splitter BARS_SPLITTER = Splitter.on("||").omitEmptyStrings();

    @Override
    public abstract boolean is(String var1);

    public abstract String getName();

    public String getSample() {
        return DEFAULT_SAMPLE;
    }

    public String toString() {
        return this.getName();
    }

    public static MatchValue of(String command) {
        String originalArg = command;
        int colonPos = command.indexOf(47);
        String subargument = null;
        if (colonPos >= 0) {
            subargument = command.substring(colonPos + 1);
            command = command.substring(0, colonPos);
        }
        try {
            MatchValue result = null;
            switch (command) {
                case "any": {
                    result = AnyMatchValue.of(subargument);
                    break;
                }
                case "set": {
                    result = SetMatchValue.of(subargument);
                    break;
                }
                case "validity": {
                    result = ValidityMatchValue.of(subargument);
                    break;
                }
                case "bcp47": {
                    result = Bcp47MatchValue.of(subargument);
                    break;
                }
                case "range": {
                    result = RangeMatchValue.of(subargument);
                    break;
                }
                case "literal": {
                    result = LiteralMatchValue.of(subargument);
                    break;
                }
                case "regex": {
                    result = RegexMatchValue.of(subargument);
                    break;
                }
                case "metazone": {
                    result = MetazoneMatchValue.of(subargument);
                    break;
                }
                case "version": {
                    result = VersionMatchValue.of(subargument);
                    break;
                }
                case "time": {
                    result = TimeMatchValue.of(subargument);
                    break;
                }
                case "or": {
                    result = OrMatchValue.of(subargument);
                    break;
                }
                case "unicodeset": {
                    result = UnicodeSpanMatchValue.of(subargument);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Illegal/Unimplemented match type: " + originalArg);
                }
            }
            if (!originalArg.equals(result.getName())) {
                System.err.println("Non-standard form or error: " + originalArg + " ==> " + result.getName());
            }
            return result;
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Problem with: " + originalArg, e);
        }
    }

    public static <T> boolean and(Predicate<T> predicate, Iterable<T> items) {
        for (T item : items) {
            if (predicate.is(item)) continue;
            return false;
        }
        return true;
    }

    public static <T> boolean or(Predicate<T> predicate, Iterable<T> items) {
        for (T item : items) {
            if (!predicate.is(item)) continue;
            return true;
        }
        return false;
    }

    public static class UnicodeSpanMatchValue
    extends MatchValue {
        final String sample;
        final UnicodeSet uset;

        public UnicodeSpanMatchValue(String key) {
            this.uset = new UnicodeSet(key);
            this.sample = new StringBuilder().appendCodePoint(this.uset.getRangeStart(0)).toString();
        }

        @Override
        public String getName() {
            return "unicodeset/" + this.uset;
        }

        public static UnicodeSpanMatchValue of(String key) {
            return new UnicodeSpanMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            return this.uset.span(item, UnicodeSet.SpanCondition.CONTAINED) == item.length();
        }

        @Override
        public String getSample() {
            return this.sample;
        }
    }

    public static class TimeMatchValue
    extends MatchValue {
        final String sample;
        final SimpleDateFormat formatter;

        public TimeMatchValue(String key) {
            this.formatter = new SimpleDateFormat(key, ULocale.ROOT);
            this.sample = this.formatter.format(new Date());
        }

        @Override
        public String getName() {
            return "time/" + this.formatter.toPattern();
        }

        public static TimeMatchValue of(String key) {
            return new TimeMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            try {
                this.formatter.parse(item);
                return true;
            }
            catch (ParseException e) {
                return false;
            }
        }

        @Override
        public String getSample() {
            return this.sample;
        }
    }

    public static class OrMatchValue
    extends MatchValue {
        final List<MatchValue> subtests;

        private OrMatchValue(Iterator<MatchValue> iterator) {
            this.subtests = ImmutableList.copyOf(iterator);
        }

        @Override
        public String getName() {
            return "or/" + Joiner.on("||").join(this.subtests);
        }

        public static OrMatchValue of(String key) {
            return new OrMatchValue(BARS_SPLITTER.splitToList(key).stream().map(k -> MatchValue.of(k)).iterator());
        }

        @Override
        public boolean is(String item) {
            for (MatchValue subtest : this.subtests) {
                if (!subtest.is(item)) continue;
                return true;
            }
            return false;
        }

        @Override
        public String getSample() {
            for (MatchValue subtest : this.subtests) {
                String result = subtest.getSample();
                if (result.equals(MatchValue.DEFAULT_SAMPLE)) continue;
                return result;
            }
            return MatchValue.DEFAULT_SAMPLE;
        }
    }

    public static class SetMatchValue
    extends MatchValue {
        final MatchValue subtest;

        public SetMatchValue(MatchValue subtest) {
            this.subtest = subtest;
        }

        @Override
        public String getName() {
            return "set/" + this.subtest.getName();
        }

        public static SetMatchValue of(String key) {
            return new SetMatchValue(MatchValue.of(key));
        }

        @Override
        public boolean is(String items) {
            List<String> splitItems = SPACE_SPLITTER.splitToList(items);
            if (new HashSet<String>(splitItems).size() != splitItems.size()) {
                throw new IllegalArgumentException("Set contains duplicates: " + items);
            }
            return SetMatchValue.and(this.subtest, splitItems);
        }

        @Override
        public String getSample() {
            return this.subtest.getSample();
        }
    }

    public static class AnyMatchValue
    extends MatchValue {
        final String key;

        public AnyMatchValue(String key) {
            this.key = key;
        }

        @Override
        public String getName() {
            return "any" + (String)(this.key == null ? "" : "/" + this.key);
        }

        public static AnyMatchValue of(String key) {
            return new AnyMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            return true;
        }
    }

    public static class MetazoneMatchValue
    extends MatchValue {
        private Set<String> valid;

        @Override
        public String getName() {
            return "metazone";
        }

        public static MetazoneMatchValue of(String key) {
            if (key != null) {
                throw new IllegalArgumentException("No parameter allowed");
            }
            return new MetazoneMatchValue();
        }

        @Override
        public synchronized boolean is(String item) {
            if (this.valid == null) {
                SupplementalDataInfo sdi = SupplementalDataInfo.getInstance();
                this.valid = sdi.getAllMetazones();
            }
            return this.valid.contains(item);
        }
    }

    public static class VersionMatchValue
    extends MatchValue {
        @Override
        public String getName() {
            return "version";
        }

        private VersionMatchValue(String key) {
        }

        public static VersionMatchValue of(String key) {
            if (key != null) {
                throw new IllegalArgumentException("No parameter allowed");
            }
            return new VersionMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            try {
                VersionInfo.getInstance(item);
            }
            catch (Exception e) {
                return false;
            }
            return true;
        }
    }

    public static class RegexMatchValue
    extends MatchValue {
        private final Pattern pattern;

        @Override
        public String getName() {
            return "regex/" + this.pattern;
        }

        private RegexMatchValue(String key) {
            this.pattern = Pattern.compile(key);
        }

        public static RegexMatchValue of(String key) {
            return new RegexMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            return this.pattern.matcher(item).matches();
        }
    }

    public static class LiteralMatchValue
    extends MatchValue {
        private final Set<String> items;

        @Override
        public String getName() {
            return "literal/" + Joiner.on(", ").join(this.items);
        }

        private LiteralMatchValue(String key) {
            this.items = ImmutableSet.copyOf(LIST.splitToList(key));
        }

        public static LiteralMatchValue of(String key) {
            return new LiteralMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            return this.items.contains(item);
        }

        @Override
        public String getSample() {
            return this.items.iterator().next();
        }
    }

    public static class RangeMatchValue
    extends MatchValue {
        private final double start;
        private final double end;
        private final boolean isInt;

        @Override
        public String getName() {
            return "range/" + (this.isInt ? (long)this.start + "~" + (long)this.end : this.start + "~" + this.end);
        }

        private RangeMatchValue(String key) {
            Iterator<String> parts = RANGE.split(key).iterator();
            this.start = Double.parseDouble(parts.next());
            this.end = Double.parseDouble(parts.next());
            boolean bl = this.isInt = !key.contains(".");
            if (parts.hasNext()) {
                throw new IllegalArgumentException("Range must be of form <int>~<int>");
            }
        }

        public static RangeMatchValue of(String key) {
            return new RangeMatchValue(key);
        }

        @Override
        public boolean is(String item) {
            double value;
            if (this.isInt && item.contains(".")) {
                return false;
            }
            try {
                value = Double.parseDouble(item);
            }
            catch (NumberFormatException e) {
                return false;
            }
            return this.start <= value && value <= this.end;
        }

        @Override
        public String getSample() {
            return String.valueOf((int)(this.start + this.end) / 2);
        }
    }

    public static class Bcp47MatchValue
    extends MatchValue {
        private final String key;
        private Set<String> valid;

        @Override
        public String getName() {
            return "bcp47/" + this.key;
        }

        private Bcp47MatchValue(String key) {
            this.key = key;
        }

        public static Bcp47MatchValue of(String key) {
            return new Bcp47MatchValue(key);
        }

        @Override
        public synchronized boolean is(String item) {
            if (this.valid == null) {
                SupplementalDataInfo sdi = SupplementalDataInfo.getInstance();
                Relation<String, String> keyToSubtypes = sdi.getBcp47Keys();
                Relation<Row.R2<String, String>, String> keySubtypeToAliases = sdi.getBcp47Aliases();
                HashMap<String, String> aliasesToKey = new HashMap<String, String>();
                for (String key : keyToSubtypes.keySet()) {
                    Set<String> aliases = keySubtypeToAliases.get(Row.of(key, ""));
                    if (aliases == null) continue;
                    for (String string : aliases) {
                        aliasesToKey.put(string, key);
                    }
                }
                switch (this.key) {
                    case "anykey": {
                        Set<String> keyList = keyToSubtypes.keySet();
                        this.valid = new TreeSet<String>(keyList);
                        for (String keyItem : keyList) {
                            this.addAliases(keySubtypeToAliases, keyItem, "");
                        }
                        this.valid.add("x");
                        break;
                    }
                    case "anyvalue": {
                        Set<String> subtypeList;
                        this.valid = new TreeSet<String>(keyToSubtypes.values());
                        for (String keyItem : keyToSubtypes.keySet()) {
                            subtypeList = keyToSubtypes.get(keyItem);
                            for (String subtypeItem : subtypeList) {
                                this.addAliases(keySubtypeToAliases, keyItem, subtypeItem);
                            }
                        }
                        this.valid.add("generic");
                        break;
                    }
                    default: {
                        String string;
                        Set<String> subtypeList = keyToSubtypes.get(this.key);
                        if (subtypeList == null && (string = (String)aliasesToKey.get(this.key)) != null) {
                            subtypeList = keyToSubtypes.get(string);
                        }
                        try {
                            this.valid = new TreeSet<String>(subtypeList);
                        }
                        catch (Exception exception) {
                            throw new IllegalArgumentException("Illegal keyValue: " + this.getName());
                        }
                        for (String subtypeItem : subtypeList) {
                            this.addAliases(keySubtypeToAliases, this.key, subtypeItem);
                        }
                        switch (this.key) {
                            case "ca": {
                                this.valid.add("generic");
                            }
                        }
                    }
                }
                this.valid = ImmutableSet.copyOf(this.valid);
            }
            return this.valid.contains(item);
        }

        private void addAliases(Relation<Row.R2<String, String>, String> keySubtypeToAliases, String keyItem, String subtype) {
            Set<String> aliases = keySubtypeToAliases.get(Row.of(keyItem, subtype));
            if (aliases != null && !aliases.isEmpty()) {
                this.valid.addAll(aliases);
            }
        }

        @Override
        public String getSample() {
            this.is("X");
            return this.valid == null ? "XX" : this.valid.iterator().next();
        }
    }

    public static class ValidityMatchValue
    extends MatchValue {
        private final StandardCodes.LstrType type;
        private final boolean shortId;
        private final Set<Validity.Status> statuses;
        private static Map<String, Validity.Status> shortCodeToStatus;
        private static final EnumParser<Validity.Status> enumParser;

        @Override
        public String getName() {
            return "validity/" + (this.shortId ? "short-" : "") + this.type.toString() + (String)(enumParser.isAll(this.statuses) ? "" : "/" + enumParser.format(this.statuses));
        }

        private ValidityMatchValue(StandardCodes.LstrType type) {
            this(type, null, false);
        }

        private ValidityMatchValue(StandardCodes.LstrType type, Set<Validity.Status> statuses, boolean shortId) {
            this.type = type;
            if (type != StandardCodes.LstrType.unit && shortId) {
                throw new IllegalArgumentException("short- not supported except for units");
            }
            this.shortId = shortId;
            this.statuses = statuses == null ? EnumSet.allOf(Validity.Status.class) : ImmutableSet.copyOf(statuses);
        }

        public static MatchValue of(String typeName) {
            boolean shortId;
            if (typeName.equals("locale")) {
                return new LocaleMatchValue();
            }
            int slashPos = typeName.indexOf(47);
            Set<Validity.Status> statuses = null;
            if (slashPos > 0) {
                statuses = enumParser.parse(typeName.substring(slashPos + 1));
                typeName = typeName.substring(0, slashPos);
            }
            if (shortId = typeName.startsWith("short-")) {
                typeName = typeName.substring(6);
            }
            StandardCodes.LstrType type = StandardCodes.LstrType.fromString(typeName);
            return new ValidityMatchValue(type, statuses, shortId);
        }

        @Override
        public boolean is(String item) {
            switch (this.type) {
                case script: {
                    if (!SCRIPT_HACK.contains(item)) break;
                    return true;
                }
                case variant: {
                    if (VARIANT_HACK.contains(item)) {
                        return true;
                    }
                    item = item.toLowerCase(Locale.ROOT);
                    break;
                }
                case language: {
                    item = item.equals("root") ? "und" : item;
                    break;
                }
                case unit: {
                    Validity.Status status;
                    if (!this.shortId) break;
                    if (shortCodeToStatus == null) {
                        TreeMap<String, Validity.Status> _shortCodeToStatus = new TreeMap<String, Validity.Status>();
                        for (Map.Entry<String, Validity.Status> entry : Validity.getInstance().getCodeToStatus(StandardCodes.LstrType.unit).entrySet()) {
                            String key = entry.getKey();
                            Validity.Status status2 = entry.getValue();
                            String shortKey = key.substring(key.indexOf(45) + 1);
                            Validity.Status old = (Validity.Status)((Object)_shortCodeToStatus.get(shortKey));
                            if (old != null) continue;
                            _shortCodeToStatus.put(shortKey, status2);
                        }
                        shortCodeToStatus = ImmutableMap.copyOf(_shortCodeToStatus);
                    }
                    return (status = shortCodeToStatus.get(item)) != null && this.statuses.contains((Object)status);
                }
            }
            Validity.Status status = Validity.getInstance().getCodeToStatus(this.type).get(item);
            return status != null && this.statuses.contains((Object)status);
        }

        @Override
        public String getSample() {
            return Validity.getInstance().getCodeToStatus(this.type).keySet().iterator().next();
        }

        static {
            enumParser = EnumParser.of(Validity.Status.class);
        }
    }

    public static class EnumParser<T extends Enum> {
        private final Class<T> aClass;
        private final Set<T> all;

        private EnumParser(Class<T> aClass) {
            this.aClass = aClass;
            this.all = ImmutableSet.copyOf(EnumSet.allOf(aClass));
        }

        public static <T> EnumParser of(Class<T> aClass) {
            return new EnumParser<T>(aClass);
        }

        public Set<T> parse(String text) {
            AbstractSet statuses = EnumSet.noneOf(this.aClass);
            boolean negative = text.startsWith("!");
            if (negative) {
                text = text.substring(1);
            }
            for (String item : SPLIT_SPACE_OR_COMMA.split(text)) {
                statuses.add(this.getItem(item));
            }
            if (negative) {
                TreeSet<T> temp = new TreeSet<T>(this.all);
                temp.removeAll(statuses);
                statuses = temp;
            }
            return ImmutableSet.copyOf(statuses);
        }

        private T getItem(String text) {
            try {
                return (T)((Enum)this.aClass.getMethod("valueOf", String.class).invoke(null, text));
            }
            catch (Exception e) {
                throw new IllegalArgumentException(e);
            }
        }

        public String format(Set<?> set) {
            if (set.size() > this.all.size() / 2) {
                TreeSet<T> temp = new TreeSet<T>(this.all);
                temp.removeAll(set);
                return "!" + Joiner.on(' ').join(temp);
            }
            return Joiner.on(' ').join(set);
        }

        public boolean isAll(Set<Validity.Status> statuses) {
            return statuses.equals(this.all);
        }
    }

    public static class LocaleMatchValue
    extends MatchValue {
        private final Predicate<String> lang;
        private final Predicate<String> script;
        private final Predicate<String> region;
        private final Predicate<String> variant;

        public LocaleMatchValue() {
            this(null);
        }

        public LocaleMatchValue(Set<Validity.Status> statuses) {
            this.lang = new ValidityMatchValue(StandardCodes.LstrType.language, statuses, false);
            this.script = new ValidityMatchValue(StandardCodes.LstrType.script, statuses, false);
            this.region = new ValidityMatchValue(StandardCodes.LstrType.region, statuses, false);
            this.variant = new ValidityMatchValue(StandardCodes.LstrType.variant, statuses, false);
        }

        @Override
        public String getName() {
            return "validity/locale";
        }

        @Override
        public boolean is(String item) {
            LanguageTagParser ltp;
            if (!item.contains("_")) {
                return this.lang.is(item);
            }
            try {
                ltp = new LanguageTagParser().set(item);
            }
            catch (Exception e) {
                return false;
            }
            return !(!this.lang.is(ltp.getLanguage()) || !ltp.getScript().isEmpty() && !this.script.is(ltp.getScript()) || !ltp.getRegion().isEmpty() && !this.region.is(ltp.getRegion()) || !ltp.getVariants().isEmpty() && !LocaleMatchValue.and(this.variant, ltp.getVariants()) || !ltp.getExtensions().isEmpty() || !ltp.getLocaleExtensions().isEmpty());
        }

        @Override
        public String getSample() {
            return "de";
        }
    }
}

