/*
 * Decompiled with CFR 0.152.
 */
package org.unicode.cldr.tool.resolver;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;
import org.unicode.cldr.tool.FilterFactory;
import org.unicode.cldr.tool.Option;
import org.unicode.cldr.tool.resolver.ResolutionType;
import org.unicode.cldr.tool.resolver.ResolverUtils;
import org.unicode.cldr.util.CLDRFile;
import org.unicode.cldr.util.CLDRPaths;
import org.unicode.cldr.util.CldrUtility;
import org.unicode.cldr.util.Factory;
import org.unicode.cldr.util.LocaleIDParser;
import org.unicode.cldr.util.SimpleXMLSource;

public class CldrResolver {
    public static final String CODE_FALLBACK = "code-fallback";
    public static final String ROOT = "root";
    private static final Option.Options options = new Option.Options("This program is used to convert CLDR XML files into their resolved versions.\nPlease refer to the following options. Options are not case sensitive.\n\texample: org.unicode.cldr.tool.resolver.CldrResolver -s xxx -d yyy -l en").add("locale", Character.valueOf('l'), ".*", ".*", "The locales to generate resolved files for").add("sourcedir", ".*", "Source directory for CLDR files").add("destdir", ".*", "Destination directory for output files").add("resolutiontype", Character.valueOf('r'), "\\w+", "simple", "The resolution type to be used").add("mindraftstatus", Character.valueOf('m'), ".*", "unconfirmed", "The minimum draft status").add("verbosity", Character.valueOf('v'), "\\d", "2", "The verbosity level for comments during generation").add("usealtvalues", Character.valueOf('a'), null, null, "Use alternate values in FilterFactory for the locale data to be resolved.").add("organization", Character.valueOf('o'), ".*", null, "Filter by this organization's coverage level");
    private Factory cldrFactory;
    private ResolutionType resolutionType;
    private final int INITIAL_RESOLVED_CACHE_SIZE = 10;
    private Cache<String, CLDRFile> resolvedCache = CacheBuilder.newBuilder().initialCapacity(10).build();

    public static void main(String[] args) {
        options.parse(args, true);
        ResolutionType resolutionType = ResolutionType.SIMPLE;
        Option option = options.get("resolutiontype");
        if (option.doesOccur()) {
            try {
                resolutionType = ResolutionType.forString(option.getValue());
            }
            catch (IllegalArgumentException e) {
                ResolverUtils.debugPrintln("Warning: " + e.getMessage(), 1);
                ResolverUtils.debugPrintln("Using default resolution type " + resolutionType.toString(), 1);
            }
        }
        String srcDir = null;
        option = options.get("sourcedir");
        srcDir = option.doesOccur() ? option.getValue() : CLDRPaths.MAIN_DIRECTORY;
        option = options.get("destdir");
        File dest = option.doesOccur() ? new File(option.getValue()) : new File(CLDRPaths.GEN_DIRECTORY, "resolver");
        if (!dest.exists()) {
            dest.mkdir();
        }
        String destDir = dest.getAbsolutePath();
        int verbosityParsed = Integer.parseInt(options.get("verbosity").getValue());
        if (verbosityParsed < 0 || verbosityParsed > 5) {
            ResolverUtils.debugPrintln("Warning: Verbosity must be between 0 and 5, inclusive.  Using default value " + ResolverUtils.verbosity, 1);
        } else {
            ResolverUtils.verbosity = verbosityParsed;
        }
        option = options.get("mindraftstatus");
        CLDRFile.DraftStatus minDraftStatus = option.doesOccur() ? CLDRFile.DraftStatus.forString(option.getValue()) : CLDRFile.DraftStatus.unconfirmed;
        Factory factory = Factory.make(srcDir, ".*", minDraftStatus);
        boolean useAltValues = options.get("usealtvalues").doesOccur();
        String org = options.get("organization").getValue();
        if (useAltValues || org != null) {
            factory = FilterFactory.load(factory, org, useAltValues);
        }
        CldrResolver resolver = new CldrResolver(factory, resolutionType);
        String localeRegex = options.get("locale").getValue();
        resolver.resolve(localeRegex, destDir);
        ResolverUtils.debugPrintln("Execution complete.", 3);
    }

    public CldrResolver(Factory factory, ResolutionType resolutionType) {
        this.cldrFactory = factory;
        this.resolutionType = resolutionType;
    }

    public void resolve(String localeRegex, File outputDir) {
        if (!outputDir.isDirectory()) {
            throw new IllegalArgumentException(outputDir.getPath() + " is not a directory");
        }
        for (String locale : this.getLocaleNames(localeRegex)) {
            ResolverUtils.debugPrintln("Processing locale " + locale + "...", 2);
            CLDRFile resolved = this.resolveLocale(locale);
            CldrResolver.printToFile(resolved, outputDir);
        }
    }

    public Set<String> getLocaleNames(String localeRegex) {
        ResolverUtils.debugPrint("Getting list of locales...", 3);
        Set<String> allLocales = this.cldrFactory.getAvailable();
        TreeSet<String> locales = new TreeSet<String>();
        for (String locale : allLocales) {
            if (locale.matches(localeRegex)) {
                locales.add(locale);
                continue;
            }
            ResolverUtils.debugPrintln("Locale " + locale + "does not match the pattern.  Skipping...\n", 4);
        }
        ResolverUtils.debugPrintln("done.\n", 3);
        return locales;
    }

    public CLDRFile resolveLocale(String locale) {
        CLDRFile base = this.cldrFactory.make(locale, true);
        CLDRFile resolved = this.resolvedCache.getIfPresent(locale);
        if (resolved != null) {
            return resolved;
        }
        ResolverUtils.debugPrintln("Processing " + locale + "...", 2);
        resolved = this.resolveLocaleInternal(base, this.resolutionType);
        this.resolvedCache.put(locale, resolved);
        return resolved;
    }

    private CLDRFile resolveLocaleInternal(CLDRFile file, ResolutionType resolutionType) {
        String locale = file.getLocaleID();
        ArrayList<CLDRFile> ancestors = new ArrayList<CLDRFile>();
        if (resolutionType == ResolutionType.SIMPLE && !locale.equals(ROOT)) {
            String parentLocale = locale;
            do {
                parentLocale = LocaleIDParser.getSimpleParent(parentLocale);
                ancestors.add(this.resolveLocale(parentLocale));
            } while (!parentLocale.equals(ROOT));
        }
        CLDRFile resolved = new CLDRFile(new SimpleXMLSource(locale));
        Set<String> basePaths = ResolverUtils.getAllPaths(file);
        for (String distinguishedPath : basePaths) {
            ResolverUtils.debugPrintln("Distinguished path: " + distinguishedPath, 5);
            if (distinguishedPath.endsWith("/alias")) {
                ResolverUtils.debugPrintln("This path is an alias.  Dropping...", 5);
                continue;
            }
            if (resolutionType == ResolutionType.NO_CODE_FALLBACK && file.getSourceLocaleID(distinguishedPath, null).equals(CODE_FALLBACK)) continue;
            String baseValue = file.getStringValue(distinguishedPath);
            if (resolutionType == ResolutionType.SIMPLE) {
                CLDRFile ancestor;
                String parentValue = null;
                Iterator iterator = ancestors.iterator();
                while (iterator.hasNext() && (parentValue = (ancestor = (CLDRFile)iterator.next()).getStringValue(distinguishedPath)) == null) {
                }
                ResolverUtils.debugPrintln("    Parent value : " + ResolverUtils.strRep(parentValue), 5);
                if (CldrResolver.areEqual(parentValue, baseValue)) continue;
            }
            ResolverUtils.debugPrintln("  Adding to resolved file.", 5);
            String path = resolutionType == ResolutionType.SIMPLE ? distinguishedPath : file.getFullXPath(distinguishedPath);
            ResolverUtils.debugPrintln("Path to be saved: " + path, 5);
            resolved.add(path, baseValue);
        }
        if (ancestors.size() > 0) {
            CLDRFile ancestor = (CLDRFile)ancestors.get(0);
            ResolverUtils.debugPrintln("Adding UNDEFINED values based on ancestor: " + ancestor.getLocaleID(), 3);
            for (String distinguishedPath : ResolverUtils.getAllPaths(ancestor)) {
                if (basePaths.contains(distinguishedPath) || ancestor.getStringValue(distinguishedPath).equals(CldrUtility.NO_INHERITANCE_MARKER)) continue;
                ResolverUtils.debugPrintln("Added UNDEFINED value for path: " + distinguishedPath, 4);
                resolved.add(distinguishedPath, CldrUtility.NO_INHERITANCE_MARKER);
            }
        }
        return resolved;
    }

    public void resolve(String localeRegex, String outputDir) {
        this.resolve(localeRegex, new File(outputDir));
    }

    private static void printToFile(CLDRFile cldrFile, File directory) {
        ResolverUtils.debugPrint("Printing file...", 2);
        try {
            PrintWriter pw = new PrintWriter(new File(directory, cldrFile.getLocaleID() + ".xml"), "UTF-8");
            cldrFile.write(pw);
            pw.close();
            ResolverUtils.debugPrintln("done.\n", 2);
        }
        catch (FileNotFoundException e) {
            ResolverUtils.debugPrintln("\nFile not found: " + e.getMessage(), 1);
            System.exit(1);
            return;
        }
        catch (UnsupportedEncodingException e) {
            ResolverUtils.debugPrintln("Your system does not support UTF-8 encoding: " + e.getMessage(), 1);
            System.exit(1);
            return;
        }
    }

    private static boolean areEqual(Object o1, Object o2) {
        if (o1 == o2) {
            return true;
        }
        if (o1 == null) {
            return false;
        }
        return o1.equals(o2);
    }
}

