/*
 * Decompiled with CFR 0.152.
 */
package org.unicode.cldr.draft.keyboard.windows;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.regex.Pattern;
import org.unicode.cldr.draft.keyboard.CharacterMap;
import org.unicode.cldr.draft.keyboard.IsoLayoutPosition;
import org.unicode.cldr.draft.keyboard.Keyboard;
import org.unicode.cldr.draft.keyboard.KeyboardBuilder;
import org.unicode.cldr.draft.keyboard.KeyboardId;
import org.unicode.cldr.draft.keyboard.KeyboardIdMap;
import org.unicode.cldr.draft.keyboard.KeycodeMap;
import org.unicode.cldr.draft.keyboard.ModifierKey;
import org.unicode.cldr.draft.keyboard.ModifierKeyCombination;

public final class KlcParser {
    public static final KeycodeMap KEYCODE_MAP = KeycodeMap.fromResource(KlcParser.class, "windows-keycodes.csv");
    public static final KeyboardIdMap KEYBOARD_ID_MAP = KeyboardIdMap.fromResource(KlcParser.class, "windows-locales.csv", KeyboardId.Platform.WINDOWS);
    private static final Splitter WHITESPACE_SPLITTER = Splitter.on(CharMatcher.whitespace()).omitEmptyStrings();
    private static final Splitter LINE_SPLITTER = Splitter.on(CharMatcher.anyOf("\n\r")).omitEmptyStrings();
    private final KeyboardBuilder builder;
    private final String klcContents;
    private final Table<String, Integer, String> virtualKeyAndIndexPlaceToLigature;
    private final Multimap<Integer, ModifierKeyCombination> indexToModifierKeyCombinations;
    private final List<Integer> modifierIndexes;
    private int shiftIndex;
    private static final ModifierKeyCombination SHIFT = ModifierKeyCombination.ofOnKeys(ImmutableSet.of(ModifierKey.SHIFT));
    private static final ModifierKeyCombination CAPSLOCK = ModifierKeyCombination.ofOnKeys(ImmutableSet.of(ModifierKey.CAPSLOCK));
    private static final ModifierKeyCombination SHIFT_CAPSLOCK = ModifierKeyCombination.ofOnKeys(ImmutableSet.of(ModifierKey.CAPSLOCK, ModifierKey.SHIFT));
    private static final Pattern LETTER_OR_DIGIT = Pattern.compile("[0-9]|[a-z]|[A-Z]");

    private KlcParser(String klcContents) {
        this.klcContents = Preconditions.checkNotNull(klcContents);
        this.builder = new KeyboardBuilder();
        this.virtualKeyAndIndexPlaceToLigature = HashBasedTable.create();
        this.indexToModifierKeyCombinations = ArrayListMultimap.create();
        this.modifierIndexes = Lists.newArrayList();
    }

    public static ImmutableList<Keyboard> parseLayout(String klcLayout) {
        KlcParser parser = new KlcParser(klcLayout);
        return parser.parse();
    }

    private ImmutableList<Keyboard> parse() {
        this.parseNameAndId();
        this.parseModifiers();
        this.parseLigatures();
        this.parseDeadkeys();
        this.parseLayout();
        return this.builder.build();
    }

    private void parseNameAndId() {
        int languageNamesIndex;
        int descriptionsIndex = this.klcContents.indexOf("\nDESCRIPTIONS");
        String section = this.klcContents.substring(descriptionsIndex, languageNamesIndex = this.klcContents.indexOf("\nLANGUAGENAMES"));
        List<String> lines = LINE_SPLITTER.splitToList(section);
        Preconditions.checkArgument(lines.size() == 2, section);
        String name = lines.get(1).substring(5).replace(" - Custom", "").trim();
        Preconditions.checkArgument(!name.isEmpty(), section);
        this.builder.addName(name);
        ImmutableCollection<KeyboardId> ids = KEYBOARD_ID_MAP.getKeyboardId(name);
        this.builder.addKeyboardIds(ids);
    }

    private void parseModifiers() {
        int shiftStateIndex = this.klcContents.indexOf("\nSHIFTSTATE");
        int layoutIndex = this.klcContents.indexOf("\nLAYOUT");
        String section = this.klcContents.substring(shiftStateIndex, layoutIndex);
        List<String> lines = LINE_SPLITTER.splitToList(section);
        for (int i = 1; i < lines.size(); ++i) {
            List<String> components = WHITESPACE_SPLITTER.splitToList(lines.get(i));
            Integer index = Integer.valueOf(components.get(0));
            HashSet<ModifierKey> modifiers = Sets.newHashSet();
            for (int j = 1; j < components.size(); ++j) {
                String token = components.get(j);
                if (token.equals("Shft")) {
                    modifiers.add(ModifierKey.SHIFT);
                    continue;
                }
                if (token.equals("Alt")) {
                    modifiers.add(ModifierKey.ALT);
                    continue;
                }
                if (!token.equals("Ctrl")) continue;
                modifiers.add(ModifierKey.CONTROL);
            }
            ImmutableSet<ModifierKey> dontCareKeys = ImmutableSet.of();
            if (modifiers.contains((Object)ModifierKey.ALT) || modifiers.contains((Object)ModifierKey.CONTROL)) {
                dontCareKeys = ImmutableSet.of(ModifierKey.CAPSLOCK);
            }
            this.modifierIndexes.add(index);
            ModifierKeyCombination combination = ModifierKeyCombination.ofOnAndDontCareKeys(modifiers, dontCareKeys);
            this.indexToModifierKeyCombinations.put(index, combination);
            if (combination.equals(SHIFT)) {
                this.shiftIndex = index;
            }
            if (!modifiers.contains((Object)ModifierKey.ALT) || !modifiers.contains((Object)ModifierKey.CONTROL)) continue;
            HashSet<ModifierKey> keys = Sets.newHashSet(modifiers);
            keys.remove((Object)ModifierKey.ALT);
            keys.remove((Object)ModifierKey.CONTROL);
            keys.add(ModifierKey.ALT_RIGHT);
            ModifierKeyCombination rAltCombination = ModifierKeyCombination.ofOnAndDontCareKeys(keys, ImmutableSet.of(ModifierKey.CAPSLOCK));
            this.indexToModifierKeyCombinations.put(index, rAltCombination);
        }
    }

    private int findLigatureEndIndex() {
        int keynameIndex = this.klcContents.indexOf("\nKEYNAME");
        keynameIndex = keynameIndex == -1 ? Integer.MAX_VALUE : keynameIndex;
        int deadkeyIndex = this.klcContents.indexOf("\nDEADKEY");
        deadkeyIndex = deadkeyIndex == -1 ? Integer.MAX_VALUE : deadkeyIndex;
        return Math.min(keynameIndex, deadkeyIndex);
    }

    private void parseLigatures() {
        int ligatureIndex = this.klcContents.indexOf("\nLIGATURE\r");
        if (ligatureIndex == -1) {
            return;
        }
        int endIndex = this.findLigatureEndIndex();
        String section = this.klcContents.substring(ligatureIndex, endIndex);
        List<String> lines = LINE_SPLITTER.splitToList(section);
        for (int i = 1; i < lines.size(); ++i) {
            String value;
            if (lines.get(i).startsWith("//")) continue;
            List<String> components = WHITESPACE_SPLITTER.splitToList(lines.get(i));
            String virtualKey = components.get(0);
            Integer modifierIndexPlace = Integer.valueOf(components.get(1));
            StringBuilder ligature = new StringBuilder();
            for (int j = 2; j < components.size() && !(value = components.get(j)).startsWith("//"); ++j) {
                ligature.append(this.convertHexValueToString(value));
            }
            String ligatureValue = ligature.toString();
            Preconditions.checkArgument(!ligatureValue.isEmpty(), section);
            this.virtualKeyAndIndexPlaceToLigature.put(virtualKey, modifierIndexPlace, ligature.toString());
        }
    }

    private int findLayoutEndIndex() {
        int deadKeyIndex = this.klcContents.indexOf("\nDEADKEY");
        deadKeyIndex = deadKeyIndex == -1 ? Integer.MAX_VALUE : deadKeyIndex;
        int keynameIndex = this.klcContents.indexOf("\nKEYNAME");
        keynameIndex = keynameIndex == -1 ? Integer.MAX_VALUE : keynameIndex;
        int ligatureIndex = this.klcContents.indexOf("\nLIGATURE");
        ligatureIndex = ligatureIndex == -1 ? Integer.MAX_VALUE : ligatureIndex;
        return Math.min(Math.min(deadKeyIndex, keynameIndex), ligatureIndex);
    }

    private void parseLayout() {
        int layoutIndex = this.klcContents.indexOf("\nLAYOUT");
        int endIndex = this.findLayoutEndIndex();
        String section = this.klcContents.substring(layoutIndex, endIndex);
        List<String> lines = LINE_SPLITTER.splitToList(section);
        for (int i = 1; i < lines.size(); ++i) {
            List<String> components;
            Integer scanCode;
            if (lines.get(i).startsWith("//") || (scanCode = Integer.valueOf((components = WHITESPACE_SPLITTER.splitToList(lines.get(i))).get(0), 16)) == 83 || scanCode == 126) continue;
            IsoLayoutPosition position = KEYCODE_MAP.getIsoLayoutPosition(scanCode);
            HashMap<Integer, CharacterMap> modifierIndexToCharacterMap = Maps.newHashMap();
            int index = 0;
            for (String output : Iterables.skip(components, 3)) {
                String character;
                if (output.equals("//")) break;
                Integer modifierIndex = this.modifierIndexes.get(index++);
                Collection<ModifierKeyCombination> combinations = this.indexToModifierKeyCombinations.get(modifierIndex);
                if (output.equals("-1")) continue;
                boolean deadKey = false;
                if (output.equals("%%")) {
                    String virtualKey = components.get(1);
                    Integer modifierIndexPlace = index - 1;
                    character = Preconditions.checkNotNull(this.virtualKeyAndIndexPlaceToLigature.get(virtualKey, modifierIndexPlace), "[Modifier: %s, VirtualKey: %s]", (Object)modifierIndexPlace, (Object)virtualKey, this.virtualKeyAndIndexPlaceToLigature);
                } else {
                    deadKey = output.endsWith("@");
                    output = deadKey ? output.substring(0, output.length() - 1) : output;
                    character = this.convertHexValueToString(output);
                }
                CharacterMap characterMap = CharacterMap.of(position, character);
                if (!deadKey && this.startsTransformSequence(character)) {
                    characterMap = characterMap.markAsTransformNo();
                }
                this.builder.addCharacterMap(combinations, characterMap);
                modifierIndexToCharacterMap.put(modifierIndex, characterMap);
            }
            String capsLock = components.get(2);
            if (capsLock.equals("1") || capsLock.equals("4") || capsLock.equals("5")) {
                if (modifierIndexToCharacterMap.containsKey(0)) {
                    this.builder.addCharacterMap(SHIFT_CAPSLOCK, (CharacterMap)modifierIndexToCharacterMap.get(0));
                }
                if (!modifierIndexToCharacterMap.containsKey(this.shiftIndex)) continue;
                this.builder.addCharacterMap(CAPSLOCK, (CharacterMap)modifierIndexToCharacterMap.get(this.shiftIndex));
                continue;
            }
            if (capsLock.equals("SGCap")) {
                String nextLine = lines.get(++i);
                List<String> nextLineComponents = WHITESPACE_SPLITTER.splitToList(nextLine);
                int nextLineIndex = 0;
                for (String output : Iterables.skip(nextLineComponents, 3)) {
                    if (output.equals("//")) break;
                    Integer modifierIndex = this.modifierIndexes.get(nextLineIndex++);
                    ModifierKeyCombination combination = Iterables.getOnlyElement(this.indexToModifierKeyCombinations.get(modifierIndex));
                    ModifierKeyCombination capsCombination = ModifierKeyCombination.ofOnKeys(ImmutableSet.copyOf(Iterables.concat(combination.onKeys(), ImmutableSet.of(ModifierKey.CAPSLOCK))));
                    if (output.equals("-1")) continue;
                    String character = this.convertHexValueToString(output);
                    CharacterMap characterMap = CharacterMap.of(position, character);
                    this.builder.addCharacterMap(capsCombination, characterMap);
                }
                continue;
            }
            if (capsLock.equals("0")) {
                if (modifierIndexToCharacterMap.containsKey(0)) {
                    this.builder.addCharacterMap(CAPSLOCK, (CharacterMap)modifierIndexToCharacterMap.get(0));
                }
                if (!modifierIndexToCharacterMap.containsKey(this.shiftIndex)) continue;
                this.builder.addCharacterMap(SHIFT_CAPSLOCK, (CharacterMap)modifierIndexToCharacterMap.get(this.shiftIndex));
                continue;
            }
            throw new IllegalArgumentException(capsLock);
        }
    }

    private void parseDeadkeys() {
        int deadkeyIndex = this.klcContents.indexOf("\nDEADKEY");
        while (deadkeyIndex != -1) {
            int nextDeadkeySection = this.klcContents.indexOf("\nDEADKEY", deadkeyIndex + 1);
            int endIndex = nextDeadkeySection == -1 ? this.klcContents.indexOf("\nKEYNAME") : nextDeadkeySection;
            String section = this.klcContents.substring(deadkeyIndex, endIndex);
            Preconditions.checkArgument(!section.isEmpty(), this.klcContents);
            this.parseDeadkeySection(this.klcContents.substring(deadkeyIndex, endIndex));
            deadkeyIndex = nextDeadkeySection;
        }
    }

    private void parseDeadkeySection(String section) {
        List<String> lines = LINE_SPLITTER.splitToList(section);
        String deadkey = this.convertHexValueToString(WHITESPACE_SPLITTER.splitToList(lines.get(0)).get(1));
        for (int i = 1; i < lines.size(); ++i) {
            List<String> components = WHITESPACE_SPLITTER.splitToList(lines.get(i));
            String combinationKey = this.convertHexValueToString(components.get(0));
            String output = this.convertHexValueToString(components.get(1));
            this.builder.addTransform(deadkey + combinationKey, output);
        }
    }

    private String convertHexValueToString(String hexValue) {
        if (LETTER_OR_DIGIT.matcher(hexValue).matches()) {
            return hexValue;
        }
        return Character.toString((char)Integer.valueOf(hexValue, 16).intValue());
    }

    private boolean startsTransformSequence(String string) {
        for (String sequence : this.builder.transformSequences()) {
            if (!sequence.startsWith(string)) continue;
            return true;
        }
        return false;
    }
}

